/*:
 * @target MZ
 * @plugindesc v1.2 ADVオートモード（ボイス終了まで待って自動送り）＋ループ音声検知で待機を回避＋無音行はテキスト長から自動待機
 * @author HS / ChatGPT
 * @url https://example.com/hs_auto_advance_voice
 * 
 * @help
 * このプラグインは、スイッチON中に「オートモード」を有効化します。
 * 仕様：
 *  - メッセージ表示が終わってポーズ状態になったら、
 *    ・SimpleVoice.jsのボイス再生がある場合：
 *        - 非ループの台詞ボイス…停止するまで待機 → 停止後に少しだけ“余韻”を足して自動送り
 *        - ループ設定のボイス…待機しません（環境音などで詰まらないよう、テキスト長ベースで自動送り）
 *    ・ボイスが無い場合：テキストの文字数から推定時間を待って自動送り
 *  - 選択肢／数値入力／アイテム選択が開いている場合は自動決定しません。
 *  - 手動で決定キー・クリックした場合は、通常通り即座に次へ進めます。
 *  - コモンイベントの入れ子（階層構造）でも動作します。
 *
 * 必須ではありませんが、Triacontane様のSimpleVoice.jsと併用すると
 * 「ボイスが鳴っている間は待つ」挙動が自然に機能します。
 * SimpleVoiceが未導入でも、テキスト長に基づくオート送りは有効です。
 *
 * 導入順の目安：PluginCommonBase.js → SimpleVoice.js → 本プラグイン
 *
 * 使い方：
 * 1) プラグインパラメータ「オートモード管理スイッチID」に任意のスイッチ番号を設定します。
 * 2) ゲーム中にそのスイッチをONにするとオートモードが有効になります（OFFで無効）。
 * 3) （任意）イベント内で SimpleVoice の「ボイスの演奏」→続けて「文章の表示」
 *    と並べると、その台詞の再生が終わったら自動で次に進みます。
 *
 * 追加コマンド：
 *  - オートモード切替（SetAutoMode）: 指定スイッチをON/OFFにします。
 *  - ボイス停止まで待機（WaitVoice）: その場で「ボイス停止まで待機」します（イベント制御用）。
 *
 * 既知の注意点：
 *  - ボイスをループ再生している場合（環境音など）は停止を検知できないため、
 *    本プラグインは「待機しない（＝テキスト長ベースで送る）」挙動に自動切替します。
 *  - メッセージ末尾に \^（即時クローズ）がある場合は、RPGツクール側の仕様で
 *    そのまま即時に閉じます。
 *
 * 利用規約：MITライセンス。商用・18禁利用・改変・再配布すべて自由です。
 * 
 * @param AutoModeSwitchId
 * @text オートモード管理スイッチID
 * @type switch
 * @desc このスイッチがONの間、オートモード（自動送り）が有効になります。
 * @default 1
 * 
 * @param WaitForVoice
 * @text ボイス終了待ちを有効にする
 * @type boolean
 * @desc SimpleVoice等でボイスが鳴っている間は待機し、停止後に自動送りします。
 * @default true
 * 
 * @param ExtraHoldAfterVoice
 * @text ボイス停止後の追い余韻（ms）
 * @type number
 * @min 0
 * @max 10000
 * @desc ボイス停止を検知してから、さらに待つ微小時間（余韻）。
 * @default 120
 * 
 * @param MsPerChar
 * @text 1文字あたりの待機（ms）
 * @type number
 * @min 1
 * @max 200
 * @desc ボイスが無い台詞の自動送りに使う基準（1文字=このミリ秒）。
 * @default 35
 * 
 * @param MinHoldMs
 * @text 最低待機時間（ms）
 * @type number
 * @min 0
 * @max 20000
 * @desc どんなに短い文でも、この時間より短くは待たない。
 * @default 400
 * 
 * @param MaxHoldMs
 * @text 最大待機時間（ms）
 * @type number
 * @min 0
 * @max 60000
 * @desc 長文でもこの時間を超えては待たない。
 * @default 4000
 * 
 * @param PunctBonusMs
 * @text 句読点ボーナス（ms）
 * @type number
 * @min 0
 * @max 5000
 * @desc 文末が句読点（。！？?!）で終わる場合に加算する余韻。
 * @default 300
 * 
 * @command SetAutoMode
 * @text オートモード切替
 * @desc オートモード管理スイッチをON/OFFします。
 * 
 * @arg on
 * @text ONにする
 * @type boolean
 * @default true
 * 
 * @command WaitVoice
 * @text ボイス停止まで待機
 * @desc （イベント制御用）ここでボイスが止まるまで待機します。
 */

(() => {
  'use strict';

  const PLUGIN_NAME = 'HS_AutoAdvanceVoice';
  const params = PluginManager.parameters(PLUGIN_NAME);
  const AutoModeSwitchId     = Number(params['AutoModeSwitchId'] || 1);
  const WaitForVoice         = params['WaitForVoice'] === 'true';
  const ExtraHoldAfterVoice  = Number(params['ExtraHoldAfterVoice'] || 120);
  const MsPerChar            = Number(params['MsPerChar'] || 35);
  const MinHoldMs            = Number(params['MinHoldMs'] || 400);
  const MaxHoldMs            = Number(params['MaxHoldMs'] || 4000);
  const PunctBonusMs         = Number(params['PunctBonusMs'] || 300);

  //-----------------------------------------------------------------------------
  // Plugin Commands (MZ)
  //-----------------------------------------------------------------------------
  PluginManager.registerCommand(PLUGIN_NAME, 'SetAutoMode', function(args) {
    const on = String(args.on) === 'true';
    if (AutoModeSwitchId > 0) {
      $gameSwitches.setValue(AutoModeSwitchId, on);
    }
  });

  PluginManager.registerCommand(PLUGIN_NAME, 'WaitVoice', function(_args) {
    this.setWaitMode('hs_voice');
  });

  //-----------------------------------------------------------------------------
  // AudioManager helpers（SimpleVoice.js連携）
  //-----------------------------------------------------------------------------
  // SimpleVoiceの複数/単一ボイス両対応。かつ「ループボイス」を検知。
  function voiceBuffers() {
    const list = [];
    if (Array.isArray(AudioManager._voiceBuffers)) {
      for (const b of AudioManager._voiceBuffers) if (b) list.push(b);
    } else if (AudioManager._voiceBuffer) {
      list.push(AudioManager._voiceBuffer);
    }
    return list;
  }

  function isBufferPlaying(buf) {
    try {
      return !!(buf.isPlaying?.() || !buf.isReady?.());
    } catch (_e) {
      return false;
    }
  }

  function isBufferLooping(buf) {
    try {
      // MZのWebAudio想定（内部に_loop / _sourceNode.loop 等がある）
      return !!(buf._loop || buf.loop || buf._sourceNode?.loop || buf._bufferSourceNode?.loop);
    } catch (_e) {
      return false;
    }
  }

  // 「再生中の非ループ音声があるか？」を返す
  AudioManager.isAnyNonLoopVoicePlaying = function() {
    const list = voiceBuffers();
    for (const b of list) {
      if (isBufferPlaying(b) && !isBufferLooping(b)) return true;
    }
    return false;
  };

  // 「再生中のループ音声があるか？」（環境音など）
  AudioManager.isAnyLoopVoicePlaying = function() {
    const list = voiceBuffers();
    for (const b of list) {
      if (isBufferPlaying(b) && isBufferLooping(b)) return true;
    }
    return false;
  };

  // 既存の「何かボイスが鳴っているか？」（後方互換）
  AudioManager.isAnyVoicePlaying = function() {
    const list = voiceBuffers();
    for (const b of list) {
      if (isBufferPlaying(b)) return true;
    }
    return false;
  };

  //-----------------------------------------------------------------------------
  // Game_Interpreter: waitMode に 'hs_voice' を追加（非ループのみ待つ）
  //-----------------------------------------------------------------------------
  const _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function() {
    if (this._waitMode === 'hs_voice') {
      if (AudioManager.isAnyNonLoopVoicePlaying()) {
        return true; // まだ待つ（非ループ音声が鳴っている）
      } else {
        this._waitMode = '';
        return false; // 解除
      }
    }
    return _Game_Interpreter_updateWaitMode.call(this);
  };

  //-----------------------------------------------------------------------------
  // Utility: テキストから制御文字を除去し、実文字数を数える
  //-----------------------------------------------------------------------------
  function stripControlCodes(text) {
    if (!text) return '';
    return text
      .replace(/\\[Vv]\[\d+\]/g, '')
      .replace(/\\[Nn]\[\d+\]/g, '')
      .replace(/\\[Pp]\[\d+\]/g, '')
      .replace(/\\[A-Z]\[\d+(?:,\d+)*\]/gi, '')
      .replace(/\\[.|!>|<^]|\\\\/g, '')
      .replace(/\x1b/g, '');
  }

  function estimateHoldMsFromText(text) {
    const clean = stripControlCodes(text || '');
    const len = Math.max(1, clean.length);
    let ms = len * MsPerChar;
    if (/[。．.!?！？]$/.test(clean)) {
      ms += PunctBonusMs;
    }
    ms = Math.max(MinHoldMs, Math.min(MaxHoldMs, ms));
    return ms;
  }

  //-----------------------------------------------------------------------------
  // Window_Message: オートモードの中核
  //-----------------------------------------------------------------------------
  const _Window_Message_initMembers = Window_Message.prototype.initMembers;
  Window_Message.prototype.initMembers = function() {
    _Window_Message_initMembers.call(this);
    this._hsAutoNextAt = null;  // performance.now() の予定時刻
    this._hsLastTextCache = '';
  };

  const _Window_Message_startMessage = Window_Message.prototype.startMessage;
  Window_Message.prototype.startMessage = function() {
    _Window_Message_startMessage.call(this);
    this._hsAutoNextAt = null;
    this._hsLastTextCache = $gameMessage ? $gameMessage.allText() : '';
  };

  const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
  Window_Message.prototype.terminateMessage = function() {
    _Window_Message_terminateMessage.call(this);
    this._hsAutoNextAt = null;
  };

  const _Window_Message_startPause = Window_Message.prototype.startPause;
  Window_Message.prototype.startPause = function() {
    _Window_Message_startPause.call(this);
    // ポーズ（文章表示が終わって待機に入った瞬間）
    this._hsAutoNextAt = null; // 再計算フラグ
  };

  // 選択肢等のサブウィンドウがアクティブか？
  Window_Message.prototype._hsAnySubWindowActive = function() {
    if (this.isAnySubWindowActive && this.isAnySubWindowActive()) return true;
    const scene = SceneManager._scene;
    if (scene) {
      if (scene._choiceListWindow && scene._choiceListWindow.active) return true;
      if (scene._numberInputWindow && scene._numberInputWindow.active) return true;
      if (scene._eventItemWindow && scene._eventItemWindow.active) return true;
    }
    return false;
  };

  // オートモード中に「次へ進む」トリガを拡張
  const _Window_Message_isTriggered = Window_Message.prototype.isTriggered;
  Window_Message.prototype.isTriggered = function() {
    // 手動入力は最優先で許可
    if (_Window_Message_isTriggered.call(this)) return true;
    // 条件：スイッチON・ポーズ中・サブウィンドウ非アクティブ
    if (!this.pause) return false;
    if (!_hsAutoEnabled()) return false;
    if (this._hsAnySubWindowActive()) return false;

    const now = performance.now();

    // --- ボイス待機ロジック ---
    // 非ループのボイスが再生中 ⇒ 待機継続
    if (WaitForVoice && AudioManager.isAnyNonLoopVoicePlaying()) {
      this._hsAutoNextAt = null;
      return false;
    }
    // ここに来た時点で「非ループ音声は鳴っていない」
    // ループ音声が鳴っている場合は、詰まらないようテキスト長で進める（待機対象にしない）

    // 初期化：予定時刻の設定
    if (this._hsAutoNextAt === null) {
      let waitMs = estimateHoldMsFromText(this._hsLastTextCache);
      // 非ループボイスの直後の余韻も付与したいので、少なくとも余韻は確保
      if (WaitForVoice) {
        waitMs = Math.max(waitMs, ExtraHoldAfterVoice);
      }
      this._hsAutoNextAt = now + Math.max(0, waitMs);
      return false;
    }

    // 規定の待機が経過したら、自動で次へ
    if (now >= this._hsAutoNextAt) {
      this._hsAutoNextAt = null;
      return true;
    }

    return false;
  };

  function _hsAutoEnabled() {
    return AutoModeSwitchId > 0 && $gameSwitches && $gameSwitches.value(AutoModeSwitchId);
  }

})();